<?php
/*--------------------------------------------------------------
   ProductVariantsDeleter.php 2021-06-11
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\ProductVariant\App\Data;

use Doctrine\DBAL\Connection;
use Exception;
use Gambio\Admin\Modules\ProductVariant\Model\Exceptions\DeletionOfProductVariantsFailed;
use Gambio\Admin\Modules\ProductVariant\Model\ValueObjects\ProductId;
use Gambio\Admin\Modules\ProductVariant\Model\ValueObjects\ProductVariantId;

/**
 * Class ProductVariantsDeleter
 * @package Gambio\Admin\Modules\ProductVariant\App\Data
 */
class ProductVariantsDeleter
{
    /**
     * @var Connection
     */
    protected $connection;
    
    
    /**
     * ProductVariantsDeleter constructor.
     *
     * @param Connection $connection
     */
    public function __construct(Connection $connection)
    {
        $this->connection = $connection;
    }
    
    
    /**
     * @param ProductVariantId ...$ids
     *
     * @throws DeletionOfProductVariantsFailed
     */
    public function deleteProductVariants(ProductVariantId ...$ids): void
    {
        try {
            $this->connection->beginTransaction();
            
            $ids = array_map(static function (ProductVariantId $id): int {
                return $id->value();
            },
                $ids);
            $this->deleteCombinationsFromProduct($ids);
            
            $this->connection->commit();
        } catch (Exception $exception) {
            $this->connection->rollBack();
            
            throw DeletionOfProductVariantsFailed::becauseOfException($exception);
        }
    }
    
    
    /**
     * @param int[] $ids
     *
     * @throws \Doctrine\DBAL\Exception
     */
    protected function deleteCombinationsFromProduct(array $ids): void
    {
        $this->connection->createQueryBuilder()
            ->delete('products_properties_combis')
            ->where('products_properties_combis_id IN (' . implode(',', $ids) . ')')
            ->execute();
        
        $this->connection->createQueryBuilder()
            ->delete('products_properties_combis_values')
            ->where('products_properties_combis_id IN (' . implode(',', $ids) . ')')
            ->execute();
        
        $this->connection->createQueryBuilder()
            ->delete('product_image_list_combi')
            ->where('products_properties_combis_id IN (' . implode(',', $ids) . ')')
            ->execute();
        
        $this->connection->createQueryBuilder()
            ->delete('products_properties_index')
            ->where('products_properties_combis_id IN (' . implode(',', $ids) . ')')
            ->execute();
    }
    
    
    /**
     * @description deletes all entries for $productId in  products_properties_admin_select
     *
     * @param ProductId $productId
     *
     * @throws \Doctrine\DBAL\Exception
     */
    public function deleteAdminSelectDataForProduct(ProductId $productId): void
    {
        $this->connection->createQueryBuilder()
            ->delete('products_properties_admin_select')
            ->where('products_id = :products_id')
            ->setParameter(':products_id', $productId->value())
            ->execute();
    }
    
    
    /**
     * @param ProductId $productId
     *
     * @throws DeletionOfProductVariantsFailed
     */
    public function deleteAllProductVariantsByProductId(ProductId $productId): void
    {
        try {
            
            $this->connection->beginTransaction();
            
            // reading all variant ids assigned to product id
            $variantIds = $this->connection->createQueryBuilder()
                ->select('products_properties_combis_id')
                ->from('products_properties_combis')
                ->where('products_id = :products_id')
                ->setParameter(':products_id', $productId->value())
                ->execute()
                ->fetchAll();
            
            $variantIds = array_map(static function(array $row): int {
                return (int)$row['products_properties_combis_id'];
            }, $variantIds);
    
            $this->deleteCombinationsFromProduct($variantIds);
    
            $this->connection->commit();
            
        } catch (Exception $exception) {
    
            $this->connection->rollBack();
    
            throw DeletionOfProductVariantsFailed::becauseOfException($exception);
        }
    }
    
    /**
     * @param ProductId $productId
     * @param int       $optionId
     *
     * @throws \Doctrine\DBAL\Exception
     */
    public function deleteAdminSelectDataForProductAndOption(ProductId $productId, int $optionId): void
    {
        $this->connection->createQueryBuilder()
            ->delete('products_properties_admin_select')
            ->where('products_id = :products_id')
            ->setParameter(':products_id', $productId->value())
            ->andWhere('properties_id = :properties_id')
            ->setParameter(':properties_id', $optionId)
            ->execute();
    }
}